package ru.bitel.bgbilling.modules.rcmts.dyn;

import bitel.billing.server.contract.bean.ContractModuleManager;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import ru.bitel.bgbilling.common.BGException;
import ru.bitel.bgbilling.kernel.container.managed.ServerContext;
import ru.bitel.bgbilling.kernel.contract.api.common.bean.Contract;
import ru.bitel.bgbilling.kernel.contract.api.common.service.ContractStatusService;
import ru.bitel.bgbilling.kernel.contract.status.common.bean.ContractStatus;
import ru.bitel.bgbilling.kernel.tariff.option.common.bean.ContractTariffOption;
import ru.bitel.bgbilling.kernel.tariff.option.common.bean.ContractTariffOptionData;
import ru.bitel.bgbilling.kernel.tariff.option.common.bean.TariffOption;
import ru.bitel.bgbilling.kernel.tariff.option.common.service.TariffOptionService;
import ru.bitel.bgbilling.modules.rcmts.server.bean.ContractDataHelper;
import ru.bitel.bgbilling.modules.rcmts.server.bean.RCConfigConstants;
import ru.bitel.bgbilling.modules.rcmts.server.bean.RCMtsRequestMethod;
import ru.bitel.bgbilling.modules.rcmts.server.bean.RCMtsResponseBuilder;
import ru.bitel.common.TimeUtils;
import ru.bitel.common.Utils;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;
import java.util.Set;

public class RCMTSResponseBuilderDefault
    implements RCMtsResponseBuilder
{
    private static final Logger logger = LogManager.getLogger( RCMTSResponseBuilderDefault.class );

    private final HttpServletRequest request;
    private final HttpServletResponse response;
    //вспомогательный класс для получения различных данных договора
    protected ContractDataHelper contractDataHelper;

    private final String[] uriParts;

    public RCMTSResponseBuilderDefault( HttpServletRequest request, HttpServletResponse response, ContractDataHelper contractDataHelper )
        throws IOException, BGException
    {
        this.request = request;
        this.response = response;
        this.contractDataHelper = contractDataHelper;
        this.uriParts = request.getRequestURI().split( "/" );
    }
    
    @Override
    public void response( RCMtsRequestMethod method )
        throws Exception
    {
        switch( method )
        {
            case GET_ACCOUNTS:
            {
                getAccounts();
                break;
            }
            case GET_AGREEMENTS:
            {
                getAgreements();
                break;
            }
            case GET_VGROUPS:
            {
                getVGroups();
                break;
            }
            case GET_TARIF_RASP:
            case DEL_TARIF_RASP:
            case TARIFS_RASP:
            {
                String requestMethod = request.getMethod();
                if ( "GET".equals( requestMethod ) )
                {
                    getTariffSchedule();
                    break;
                }
                else if ( "POST".equals( requestMethod ) )
                {
                    setTariffSchedule();
                    break;
                }
                else if ( "DELETE".equals( requestMethod ) )
                {
                    deleteTariffSchedule();
                    break;
                }
            }
            case BLOCK_VGROUPS:
            {
                blockVGroups();
                break;
            }
        }
    }

    @Override
    public void getAccounts()
        throws IOException, BGException
    {
        logger.info( "getAccounts" );
        try
        {
            String fio = contractDataHelper.getFio();
            String lastName = contractDataHelper.getLastName();
            String firstName = contractDataHelper.getFirstName();
            String middleName = contractDataHelper.getMiddleName();

            JSONObject responseJson = new JSONObject();
            if (contractDataHelper.getContract().getPersonType() == 1)
            {
                responseJson.put("detail", " Подключение услуги для юридического лица невозможно.");
                this.response.setStatus(405);
            }
            else
            {
                responseJson.put( "uid", contractDataHelper.getContractId() );
                responseJson.put( "name", fio );
                responseJson.put( "abonentname", firstName );
                responseJson.put( "abonentsurname", lastName );
                responseJson.put( "abonentpatronymic", middleName );
                responseJson.put( "address", contractDataHelper.getAddress() );

                JSONArray agreements = new JSONArray();
                JSONObject agreementsJson = new JSONObject();
                agreementsJson.put( "agrmid", contractDataHelper.getContractId() );

                // т.к. у себя МТС держит номер договора со своим префиксом, то приклеиваем его перед ответом
                String accountPrefix = contractDataHelper.getModuleSetup().get( RCConfigConstants.MTS_ACCOUNT_PREFIX.getKey() );
                agreementsJson.put( "number", Utils.maskBlank( accountPrefix, "" ) + contractDataHelper.getContract().getTitle() );

                agreementsJson.put( "balance", contractDataHelper.getBalance() );
                agreementsJson.put( "uid", contractDataHelper.getContractId() );

                agreements.put( agreementsJson );
                responseJson.put( "agreements", agreements );
            }

            logger.info( "Response >>> " + responseJson );
            writeJsonToResponse( responseJson );
        }
        catch( BGException ex )
        {
            response.setStatus( 400 );
            throw ex;
        }
    }

    @Override
    public void getAgreements()
        throws IOException, BGException
    {
        logger.info( "getAgreements" );

        JSONObject responseJson = new JSONObject();
        responseJson.put( "uid", contractDataHelper.getContractId() );
        responseJson.put( "agrmid", contractDataHelper.getContractId() );
        responseJson.put( "PaymentMethod", Utils.parseInt( contractDataHelper.getModuleSetup().get( RCConfigConstants.PAYMENT_METHOD.getKey() ), 2 ) );

        logger.info( "Response >>> " + responseJson );
        writeJsonToResponse( responseJson );
    }

    @Override
    public void getVGroups()
        throws Exception
    {
        logger.info( "getVGroups" );
        Contract contract = contractDataHelper.getContract();

        JSONObject responseJson = new JSONObject();
        JSONArray vgroups = new JSONArray();

        JSONObject innerJson1 = new JSONObject();
        innerJson1.put( "vgid", contractDataHelper.getContractId() );
        innerJson1.put( "agrmid", contractDataHelper.getContractId() );
        innerJson1.put( "uid", contractDataHelper.getContractId() );

        ServerContext serverContext = ServerContext.get();
        List<ContractTariffOption> options = getMTSTariffOptions( serverContext.getService( TariffOptionService.class, 0 ), contract.getId() );
        boolean convergentIsEnable = Utils.notEmptyCollection( options );
        int optionId = convergentIsEnable ? options.get( 0 ).getOptionId() : 1;
        String optionTitle = "Не конвергентный тариф";
        if ( convergentIsEnable )
        {
            TariffOption tariffOption = serverContext.getService( TariffOptionService.class, 0 ).tariffOptionGet( optionId );
            if ( tariffOption != null )
            {
                optionTitle = tariffOption.getTitle();
            }
        }

        innerJson1.put( "tarid", optionId );
        innerJson1.put( "login", contract.getTitle() );
        innerJson1.put( "blocked", contractDataHelper.contractIsActive() ? 0 : 3 ); //0 - учетка активна, 3 - заблокирована. (посмотреть в каких статусах в биллинге договор является активным и проставить 0 или 3 взависимости от статуса)
        innerJson1.put( "packagetypeid", definePackageTypeId(serverContext, contract.getId()) ); //Тип услуги согласно справочнику в документации

        innerJson1.put( "tarifdescr", optionTitle );
        innerJson1.put( "accondate", TimeUtils.format( contract.getDateFrom(), TimeUtils.DATE_FORMAT_PATTERN_YYYY_MM_DD_HHMMSS) ); //Время создания(первого включения) учетной записи
        vgroups.put( innerJson1 );

        responseJson.put( "vgroups", vgroups );

        logger.info( "Response >>> " + responseJson );
        writeJsonToResponse( responseJson );
    }

    private int definePackageTypeId( ServerContext serverContext, int contractId )
        throws BGException
    {
        int packageTypeId = 0;
        Set<Integer> inetMids = Utils.toIntegerSet( contractDataHelper.getModuleSetup().get( RCConfigConstants.INET_MODULES_IDS.getKey() ) );
        Set<Integer> tvMids = Utils.toIntegerSet( contractDataHelper.getModuleSetup().get( RCConfigConstants.TV_MODULES_IDS.getKey() ) );

        ContractModuleManager contractModuleManager = new ContractModuleManager( serverContext.getConnection() );
        Set<Integer> modulesOnContract = contractModuleManager.getContractModuleSet( contractId );
        boolean hasInetModule = inetMids.stream().anyMatch( modulesOnContract::contains );
        boolean hasTvModule = tvMids.stream().anyMatch( modulesOnContract::contains );
        //если есть модуль инет, то 1
        //если есть и тв тоже, то 3
        //если только тв, то 2

        // взято из справочника предоставленного МТС - "справочник packagetypeid"
        // вряд ли когда-то это будет меняться, поэтому не стал выносить в enum
        if ( hasInetModule && hasTvModule )
        {
            packageTypeId = 16385;
        }
        else if ( hasInetModule )
        {
            packageTypeId = 1;
        }
        else if ( hasTvModule )
        {
            packageTypeId = 16384;
        }

        return packageTypeId;
    }

    @Override
    public void getTariffSchedule()
        throws IOException
    {
        logger.info( "getTariffSchedule" );

        JSONArray response = new JSONArray();
        JSONObject innerJson = new JSONObject();
        innerJson.put( "recordid", 0 );
        response.put( innerJson );

        logger.info( "Response >>> " + response );
        writeJsonArrayToResponse( response );
    }
    
    @Override
    public void deleteTariffSchedule()
        throws IOException
    {
        logger.info( "deleteTariffSchedule" );

        int tariffOptionID = Utils.parseInt( uriParts[7], -1 );
        if ( tariffOptionID < 0 )
        {
            logger.error( "Не удалось получить договор или опцию из запроса" );
            response.sendError( 500, "Не удалось получить договор или опцию из запроса" );
            return;
        }

        logger.debug( "Запрос на удаление запланированной установки ТО id=" + tariffOptionID );

        JSONObject response = new JSONObject();
        response.put( "status", "OK" );

        logger.info( "Response >>> " + response );
        writeJsonToResponse( response );
    }

    @Override
    public void setTariffSchedule()
        throws IOException, BGException
    {
        logger.info( "setTariffSchedule" );

        int contractId = contractDataHelper.getContractId();
        int tariffOptionId = Utils.parseInt( uriParts[7], -1 );
        if ( tariffOptionId < 0 )
        {
            String errorText = "Не удалось получить договор или опцию из запроса";
            logger.error( errorText );
            response.sendError( 404, errorText );
            return;
        }

        ServerContext serverContext = ServerContext.get();
        TariffOptionService tariffOptionService = serverContext.getService( TariffOptionService.class, 0 );

        try
        {
            if ( tariffOptionId == 1 )
            {
                // тогда развал пакета мтс
                deactivateMtsOptionsOnContract( tariffOptionService, contractId );
            }
            else if ( tariffOptionId > 1 )
            {
                try
                {
                    TariffOption tariffOption = tariffOptionService.tariffOptionGet( tariffOptionId );
                    int activateModeId = Utils.notEmptyCollection( tariffOption.getActivateModeList() ) ? tariffOption.getActivateModeList().get( 0 ).getId() : -1;
                    if ( activateModeId < 0 )
                    {
                        throw new BGException( "Не удалось получить ID режима активации для тарифной опции ID=" + tariffOptionId );
                    }

                    deactivateMtsOptionsOnContract( tariffOptionService, contractId );

                    logger.info( "Активация ТО с ID={}, с режимом активации ID={}, для contractId={}", tariffOptionId, activateModeId, contractId );
                    
                    ContractTariffOptionData optionActivate = new ContractTariffOptionData();
                    optionActivate.setContractId( contractId );
                    optionActivate.setOptionId( tariffOptionId );
                    optionActivate.setModeId( activateModeId );
                    
                    tariffOptionService.contractTariffOptionActivate( optionActivate );
                }
                catch( NullPointerException npe )
                {
                    throw new BGException( "Не удалось найти тарифную опцию с ID=" + tariffOptionId );
                }
            }
        }
        catch( Exception ignored ) {}
        finally
        {
            serverContext.commit();
        }

        JSONObject response = new JSONObject();
        response.put( "recordid", "1" );
        this.response.setStatus( 200 );

        logger.info( "Response >>> " + response );
        writeJsonToResponse( response );
    }

    private void deactivateMtsOptionsOnContract( TariffOptionService tariffOptionService, int contractId )
        throws Exception
    {
        Set<Integer> optionIds = Utils.toIntegerSet( contractDataHelper.getModuleSetup().get( RCConfigConstants.MTS_TARIFF_OPTIONS.getKey() ) );
        List<ContractTariffOption> contractTariffOptions = tariffOptionService.contractTariffOptionList( contractId, new Date() );
        for ( ContractTariffOption option : contractTariffOptions )
        {
            if ( optionIds.contains( option.getOptionId() ) && (option.getDeactivatedTime() == null || option.getDeactivatedTime().after( new Date() )) )
            {
                logger.info( "Деактивация опции МТС = " + option.getOptionTitle() );
                try
                {
                    tariffOptionService.contractTariffOptionDeactivate( contractId, option.getId() );
                }
                catch( Exception ex )
                {
                    logger.error( ex );
                }
            }
        }
    }

    @Override
    public void blockVGroups()
        throws Exception
    {
        logger.info( "blockVGroups" );

        String state = uriParts[8];
        if ( Utils.isBlankString( state ) )
        {
            String error = "Ошибка получения параметров запроса";
            logger.error( error );
            response.sendError( 400, error );
            return;
        }

        JSONObject response = new JSONObject();

        switchContractStatus( contractDataHelper.getContract(), state );

        response.put( "status", "OK" );
        this.response.setStatus( 200 );

        logger.info( "Response >>> " + response );
        writeJsonToResponse( response );
    }

    /**
     *
     * @param contractId id договора
     * @return список тарифных опций на договоре, который относится к МТС и был заполнен ранее в конфиге модуля
     */
    private List<ContractTariffOption> getMTSTariffOptions( TariffOptionService tariffOptionService, int contractId )
        throws Exception
    {
        String optionIds = contractDataHelper.getModuleSetup().get( RCConfigConstants.MTS_TARIFF_OPTIONS.getKey(), "" );
        if ( Utils.isBlankString( optionIds ) )
            throw new BGException("В конфигурации модуля не заполнены id тарифных опций!");

        List<Integer> optionIds0 = Utils.toIntegerList( optionIds );
        List<ContractTariffOption> options = tariffOptionService.contractTariffOptionList( contractId, new Date() );
        options.removeIf( o -> !optionIds0.contains( o.getOptionId() ) );
        return options;
    }

    /**
     * Взависимости от переданного состояние меняется статус договора на активеный или заблокированный.
     * Активные и заблокированные статусы договора указывается в конфигурации модуля.
     */
    private void switchContractStatus( Contract contract, String state )
        throws Exception
    {
        int activeStatus = contractDataHelper.getModuleSetup().getInt( RCConfigConstants.ACTIVE_STATUS.getKey(), -1 );
        int blockStatus = contractDataHelper.getModuleSetup().getInt( RCConfigConstants.BLOCK_STATUS.getKey(), -1 );
        if ( activeStatus < 0 || blockStatus < 0 )
        {
            throw new BGException("В конфигурации модуля не установлены статусы");
        }
        ServerContext serverContext = ServerContext.get();
        ContractStatusService contractStatusService = serverContext.getService( ContractStatusService.class, 0 );

        int status = "on".equals( state ) ? blockStatus : activeStatus;

        ContractStatus contractStatus = new ContractStatus();
        contractStatus.setStatus( status );
        contractStatus.setContractId( contract.getId() );
        contractStatus.setDateFrom( new Date() );
        contractStatus.setComment( "Установлено запросом от МТС" );

        contractStatusService.contractStatusChange( contract.getId(), contractStatus, false, false );
        serverContext.commit();
    }

    protected void writeJsonToResponse( JSONObject json )
        throws IOException
    {
        response.getOutputStream().write( json.toString().getBytes(StandardCharsets.UTF_8) );
    }

    protected void writeJsonArrayToResponse( JSONArray jsonArray )
        throws IOException
    {
        response.getOutputStream().write( jsonArray.toString().getBytes(StandardCharsets.UTF_8) );
    }

    protected JSONObject getJSONBody( HttpServletRequest request )
        throws IOException
    {
        JSONObject result;
        try
        {
            result = new JSONObject( getRequestBody( request ) );
        }
        catch( JSONException ex )
        {
            result = new JSONObject();
        }
        return result;
    }

    protected String getRequestBody( HttpServletRequest request )
    {
        StringBuilder jsonString = new StringBuilder();
        try (BufferedReader br = new BufferedReader( new InputStreamReader( request.getInputStream() ) ))
        {
            String line;
            while( (line = br.readLine()) != null )
            {
                jsonString.append( line );
            }
        }
        catch( IOException e )
        {
            throw new RuntimeException( e );
        }
        return jsonString.toString();
    }
}