package ru.bitel.bgbilling.modules.megafon.dyn.tariff;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.bitel.bgbilling.kernel.container.managed.ServerContext;
import ru.bitel.bgbilling.kernel.tariff.server.bean.DynamicNodeAdapter;
import ru.bitel.bgbilling.kernel.tariff.server.tree.AbstractTariffRequest;
import ru.bitel.bgbilling.kernel.tariff.server.tree.ServiceCost;
import ru.bitel.bgbilling.kernel.tariff.server.tree.TariffContext;
import ru.bitel.bgbilling.modules.megafon.common.bean.MegafonOption;
import ru.bitel.bgbilling.modules.megafon.common.bean.MegafonOptionType;
import ru.bitel.bgbilling.modules.megafon.common.bean.MegafonProduct;
import ru.bitel.bgbilling.modules.megafon.common.service.MegafonService;
import ru.bitel.bgbilling.modules.npay.server.tariff.NPayTariffRequest;
import ru.bitel.bgbilling.server.util.Setup;
import ru.bitel.common.TimeUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

public class MegafonDynTariff
    extends DynamicNodeAdapter
{
    private static final Logger log = LogManager.getLogger();

    private static final int MEGAFON_MID = 10;

    @Override
    public boolean execute( Setup setup, AbstractTariffRequest<? extends ServiceCost> req, TariffContext ctx )
        throws Exception
    {
        NPayTariffRequest r = null;
        if ( req instanceof NPayTariffRequest )
        {
            r = (NPayTariffRequest)req;
        }
        if ( r == null )
        {
            return false;
        }

        //ID Услуги NPAY, которая привязана к тарифу или опции Модуля Мегафон
        int npayServiceId = r.getServiceObject().getServiceId();
        ServerContext context = ServerContext.get();
        if ( !MegafonServicesCache.INSTANCE.isMegafonService( context, npayServiceId ) )
            return false;

        int contractId = r.contractId;

        LocalDate serviceStartDate = TimeUtils.convertDateToLocalDate( r.getServiceObject().getDate1() );
        LocalDate now = LocalDate.now();

        // полученная стоимость продукта или опции
        BigDecimal cost;

        MegafonProduct product = MegafonServicesCache.INSTANCE.serivcesWithProductsMap.get( npayServiceId );
        BigDecimal price = null;
        if ( product != null )
        {
            price = new BigDecimal( product.getPrice() );
            log.info( "Услуга NPAY={} определена как MegafonProductId={}, price={}", npayServiceId, product.getId(), price );
        }

        MegafonOption option = MegafonServicesCache.INSTANCE.serivcesWithOptionMap.get( npayServiceId );
        if ( option != null )
        {
            if ( option.getOptionType().equals( MegafonOptionType.ONE_TIME ) )
            {
                log.warn( "Доп.опции с типом \"доп.пакет\" - не пролонгируются! contractId={}, serviceId={}, contractId={}", contractId, npayServiceId, contractId );
            }
            else
            {
                price = option.getPrice();
                log.info( "Услуга NPAY={} определена как MegafonOptionId={}, price={}, contractId={}", npayServiceId, option.getId(), price, contractId );
            }
        }

        log.error( "Не удалось получить стоимость услуги ID={}", npayServiceId );

        if ( price != null )
        {
            // единственный вариант при котором стоимость услуги высчитывается - это если услуга началась в этом месяце НЕ 1-го числа.
            // тогда считается: price / 30 * (кол-во дней оставшихся в текущем месяце)
            if ( serviceStartDate.getDayOfMonth() > 1 && serviceStartDate.getYear() == now.getYear() && serviceStartDate.getMonth() == now.getMonth() )
            {
                int remainingDays = serviceStartDate.lengthOfMonth() - serviceStartDate.getDayOfMonth() + 1;
                cost = price.divide( BigDecimal.valueOf( 30 ), 2, RoundingMode.HALF_UP ).multiply( BigDecimal.valueOf( remainingDays ) );
                log.debug( "Стоимость высчитывается для оставшихся дней в месяце. Осталось дней={}, price={}, RESULT={}", remainingDays, price, cost );
            }
            else
            {
                cost = price;
            }

            log.info( "ResultCost={}, npayServiceId={}, contractId={}", cost, npayServiceId, contractId );

            r.serviceCost.setCost( cost );

            return true;
        }

        return false;
    }

    private static final class MegafonServicesCache
    {
        // прогрев кэша, если превышено указанное кол-во минут
        private static final int RELOAD_DELAY_IN_MINUTES = 30;

        private static final MegafonServicesCache INSTANCE;

        static
        {
            INSTANCE = new MegafonServicesCache();
        }

        private final Map<Integer, MegafonProduct> serivcesWithProductsMap = new HashMap<>();
        private final Map<Integer, MegafonOption> serivcesWithOptionMap = new HashMap<>();

        //время последнего обращения к классу
        private LocalDateTime lastCallTime;

        private boolean isMegafonService( ServerContext context, int serviceId )
            throws Exception
        {
            reloadIfNeed( context );
            return serivcesWithProductsMap.containsKey( serviceId ) || serivcesWithOptionMap.containsKey( serviceId );
        }

        private synchronized void reloadIfNeed( ServerContext context )
            throws Exception
        {
            if ( lastCallTime == null )
            {
                reload( context );
                lastCallTime = LocalDateTime.now();
                return;
            }

            if ( Duration.between( lastCallTime, LocalDateTime.now() ).toMinutes() > RELOAD_DELAY_IN_MINUTES )
            {
                reload( context );
                lastCallTime = LocalDateTime.now();
            }
        }

        private synchronized void reload( ServerContext context )
            throws Exception
        {
            this.serivcesWithProductsMap.clear();
            this.serivcesWithOptionMap.clear();

            MegafonService megafonService = context.getService( MegafonService.class, MEGAFON_MID );
            serivcesWithProductsMap.putAll( megafonService.getProductList()
                                                          .stream()
                                                          .collect( Collectors.toMap( MegafonProduct::getNpayServiceId, Function.identity() ) ) );

            serivcesWithOptionMap.putAll( megafonService.getOptionList()
                                                        .stream()
                                                        .collect( Collectors.toMap( MegafonOption::getNpayServiceId, Function.identity() ) ) );
        }
    }
}
